/*
 * File: ChaosGame.java
 * ================================================================
 * An implementation of the Chaos Game.  The program picks three
 * points to act as vertices of a triangle (these will be the
 * upper-left, lower-right, and lower-left corners of the window).
 * Starting with the top point, we repeatedly do the following:
 * 
 * 1. Choose a random one of the three corners.
 * 2. Move halfway between our current location and that corner.
 * 3. Put a dot at our location.
 * 
 * The result is pretty surprising!
 */
import acm.graphics.*;
import acm.program.*;
import acm.util.*;
import java.awt.*;
import java.awt.event.*;

public class ChaosGame extends GraphicsProgram {
	/* The amount to pause between frames. */
	private static final double PAUSE_TIME = 0.5;

	/**
	 * Runs the chaos game.
	 */
	public void run() {
		/* For elegance, let's make the background black. */
		setBackground(Color.BLACK);
		
		/* Track our current point; initially it's the upper-left
		 * corner.
		 */
		GPoint pt = new GPoint(0, 0);
		
		/* Keep playing the game. */
		while (true) {
			moveRandomly(pt);
			plotPixel(pt);
			pause(PAUSE_TIME);
		}
	}
	
	/**
	 * Moves the point randomly in accordance with the rules of
	 * the chaos game.
	 * 
	 * @param pt The point to move.
	 */
	private void moveRandomly(GPoint pt) {
		/* Choose a destination to move to. */
		GPoint destination = chooseRandomPoint();
		
		/* Move halfway to the destination. */
		pt.setLocation((pt.getX() + destination.getX()) / 2.0,
				       (pt.getY() + destination.getY()) / 2.0);
	}
	
	/**
	 * Chooses a random corner point to move to.
	 * 
	 * @return A random corner point.
	 */
	private GPoint chooseRandomPoint() {
		/* Choose one of the three corners at random. */
		RandomGenerator rgen = RandomGenerator.getInstance();
		int random = rgen.nextInt(1, 3);
		
		/* Upper-left corner. */
		if (random == 1) {
			return new GPoint(0, 0);
		}
		/* Lower-right corner. */
		else if (random == 2) {
			return new GPoint(getWidth(), getHeight());
		}
		/* Lower-left corner. */
		else {
			return new GPoint(0, getHeight());
		}
	}
	
	/**
	 * Plots a pixel at the given location with a random color.
	 * 
	 * @param pt The location to draw at.
	 */
	private void plotPixel(GPoint pt) {
		/* Create a 1x1 rectangle for our pixel. */
		GRect pixel = new GRect(pt.getX(), pt.getY(), 1, 1);
		pixel.setFilled(true);
		
		/* Choose a random color for the pixel. */
		RandomGenerator rgen = RandomGenerator.getInstance();
		pixel.setColor(rgen.nextColor());
		
		add(pixel);
	}
}
